<?php

namespace App\Http\Controllers;

use App\Models\Pelatihan;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Models\RegistrasiPeserta;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use App\Imports\RegistrasiPesertaImport;
use Illuminate\Support\Facades\Storage as FacadesStorage;

class RegistrasiPesertaController extends Controller
{

    public function index(Request $request)
    {
        // Query utama
        $query = RegistrasiPeserta::query();

        /**
         * 🔍 PENCARIAN (Nama / NIK / Email)
         */
        if ($request->filled('search')) {
            $search = $request->search;

            $query->where(function ($q) use ($search) {
                $q->where('nama', 'like', "%{$search}%")
                    ->orWhere('nik', 'like', "%{$search}%")
                    ->orWhere('email_pribadi', 'like', "%{$search}%");
            });
        }

        /**
         * 🎓 FILTER BERDASARKAN PELATIHAN
         */
        if ($request->filled('pelatihan')) {
            $query->where('nama_kegiatan', $request->pelatihan);
        }

        /**
         * 📄 DATA REGISTRASI (PAGINATION)
         */
        $registrasi = $query
            ->latest()
            ->paginate(10);

        /**
         * 📌 LIST PELATIHAN (UNTUK DROPDOWN FILTER)
         */
        $pelatihanList = RegistrasiPeserta::select('nama_kegiatan')
            ->whereNotNull('nama_kegiatan')
            ->distinct()
            ->orderBy('nama_kegiatan')
            ->pluck('nama_kegiatan');

        return view('admin.registrasi.index', compact(
            'registrasi',
            'pelatihanList'
        ));
    }


    public function create(Request $request)
    {
        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        $kabupatenOptions = [
            'Jakarta Barat',
            'Jakarta Pusat',
            'Jakarta Selatan',
            'Jakarta Timur',
            'Jakarta Utara',
            'Kepulauan Seribu',
        ];

        $agamaOptions = ['Islam', 'Kristen', 'Katolik', 'Hindu', 'Buddha', 'Konghucu', 'Lainnya'];

        return view('portal.registrasi.create', compact(
            'pelatihanList',
            // 'kabupatenOptions',
            'agamaOptions'
        ));
    }

    public function store(Request $request)
    {
        $userId = Auth::id();

        $rules = [
            'pelatihan_id' => ['required', 'exists:pelatihan,id'],

            'nama'          => ['required', 'string', 'max:255'],
            'nik'           => [
                'required',
                'regex:/^\d+$/',
                'max:50',
                // cegah nik daftar di pelatihan yang sama
                Rule::unique('registrasi_peserta', 'nik')->where(function ($q) use ($request) {
                    return $q->where('pelatihan_id', $request->pelatihan_id);
                }),
            ],
            'jenis_kelamin' => ['required', 'in:Laki-laki,Perempuan'],
            'no_hp'         => ['required', 'regex:/^\d+$/', 'max:30'],
            'email_pribadi' => ['required', 'email', 'max:255'],

            'agama' => ['required', Rule::in(['Islam', 'Kristen', 'Katolik', 'Hindu', 'Buddha', 'Konghucu', 'Lainnya'])],

            'nip'              => ['nullable', 'regex:/^\d+$/', 'max:50'],
            'jabatan'          => ['nullable', 'string', 'max:150'],
            'pangkat_golongan' => ['nullable', 'string', 'max:150'],
            'tempat_lahir'     => ['nullable', 'string', 'max:150'],
            'tanggal_lahir'    => ['nullable', 'date'],

            'pendidikan_terakhir' => ['nullable', 'string', 'max:100'],
            'prodi'               => ['nullable', 'string', 'max:150'],

            // UNIT KERJA (wajib semua)
            'unit_kerja'        => ['required', 'string', 'max:255'],
            'npsn'              => ['required', 'regex:/^\d+$/', 'max:30'],
            'alamat_unit_kerja' => ['required', 'string'],
            'kab_kota_unit'     => ['required', 'string', 'max:100'],
            'provinsi_unit'     => ['required', 'string', 'max:100'],
            'kode_pos_unit'     => ['required', 'regex:/^\d+$/', 'max:10'],
            'telp_instansi'     => ['required', 'regex:/^\d+$/', 'max:30'],

            // ALAMAT RUMAH (wajib semua)
            'alamat_rumah'    => ['required', 'string'],
            'kab_kota_rumah'  => ['required', 'string', 'max:100'],
            'provinsi_rumah'  => ['required', 'string', 'max:100'],
            'kode_pos_rumah'  => ['required', 'regex:/^\d+$/', 'max:10'],

            // NPWP & BANK (wajib)
            'npwp'        => ['required', 'regex:/^\d+$/', 'max:50'],
            'bank'        => ['required', 'string', 'max:100'],
            'no_rekening' => ['required', 'regex:/^\d+$/', 'max:50'],

            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ];

        $messages = [
            'pelatihan_id.required' => 'Pelatihan wajib dipilih.',
            'pelatihan_id.exists'   => 'Pelatihan tidak ditemukan.',

            'nama.required' => 'Nama wajib diisi.',
            'nik.required'  => 'NIK wajib diisi.',
            'nik.regex'     => 'NIK hanya boleh angka.',
            'nik.unique'    => 'NIK ini sudah terdaftar di pelatihan yang sama.',

            'jenis_kelamin.required' => 'Jenis kelamin wajib dipilih.',
            'no_hp.required' => 'No HP wajib diisi.',
            'no_hp.regex'    => 'No HP hanya boleh angka.',
            'email_pribadi.required' => 'Email pribadi wajib diisi.',
            'email_pribadi.email' => 'Format email pribadi tidak valid.',

            'agama.required' => 'Agama wajib dipilih.',

            'nip.regex' => 'NIP hanya boleh angka.',

            'unit_kerja.required' => 'Unit kerja wajib diisi.',
            'npsn.required' => 'NPSN wajib diisi.',
            'alamat_unit_kerja.required' => 'Alamat unit kerja wajib diisi.',
            'kab_kota_unit.required' => 'Kab/Kota unit kerja wajib diisi.',
            'provinsi_unit.required' => 'Provinsi unit kerja wajib diisi.',
            'kode_pos_unit.required' => 'Kode pos unit kerja wajib diisi.',
            'kode_pos_unit.regex' => 'Kode pos unit kerja hanya boleh angka.',
            'telp_instansi.required' => 'Telp instansi wajib diisi.',
            'telp_instansi.regex' => 'Telp instansi hanya boleh angka.',

            'alamat_rumah.required' => 'Alamat rumah wajib diisi.',
            'kab_kota_rumah.required' => 'Kab/Kota alamat rumah wajib diisi.',
            'provinsi_rumah.required' => 'Provinsi alamat rumah wajib diisi.',
            'kode_pos_rumah.required' => 'Kode pos alamat rumah wajib diisi.',
            'kode_pos_rumah.regex' => 'Kode pos alamat rumah hanya boleh angka.',

            'npwp.required' => 'NPWP wajib diisi.',
            'npwp.regex'    => 'NPWP hanya boleh angka.',
            'bank.required' => 'Nama bank wajib diisi.',
            'no_rekening.required' => 'No rekening wajib diisi.',
            'no_rekening.regex'    => 'No rekening hanya boleh angka.',
        ];

        $data = $request->validate($rules, $messages);

        // set "nama_kegiatan" dari tabel pelatihan (biar konsisten)
        $pelatihan = Pelatihan::findOrFail($data['pelatihan_id']);
        $data['nama_kegiatan'] = $pelatihan->nama_pelatihan;

        // set user_id untuk anti duplikat by akun
        $data['user_id'] = $userId;

        // anti duplikat user_id + pelatihan_id (biar ga meledak QueryException)
        $sudah = RegistrasiPeserta::where('user_id', $userId)
            ->where('pelatihan_id', $data['pelatihan_id'])
            ->exists();

        if ($sudah) {
            return back()
                ->withErrors(['pelatihan_id' => 'Kamu sudah pernah registrasi untuk pelatihan ini.'])
                ->withInput();
        }

        if ($request->hasFile('foto')) {
            $data['foto'] = $request->file('foto')->store('registrasi_peserta', 'public');
        }

        RegistrasiPeserta::create($data);

        return redirect()
            ->route('portal.registrasi.create')
            ->with('success', 'Registrasi berhasil disimpan.');
    }

    public function success()
    {
        return view('portal.registrasi.success');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => ['required', 'file', 'mimes:xlsx,xls', 'max:5120'],
        ]);

        try {
            Excel::import(new RegistrasiPesertaImport, $request->file('file'));

            return redirect()
                ->route('admin.registrasi.index')
                ->with('success', 'Import data registrasi dari file Excel berhasil.');
        } catch (\Throwable $e) {
            return redirect()
                ->route('admin.registrasi.index')
                ->with('error', 'Gagal import data registrasi: ' . $e->getMessage());
        }
    }

    public function show(RegistrasiPeserta $registrasi)
    {
        return view('admin.registrasi.show', [
            'registrasiPeserta' => $registrasi
        ]);
    }

    public function edit(RegistrasiPeserta $registrasi)
    {
        $pelatihanList = Pelatihan::orderBy('nama_pelatihan')
            ->get(['id', 'nama_pelatihan']);

        $agamaOptions = [
            'Islam',
            'Kristen',
            'Katolik',
            'Hindu',
            'Buddha',
            'Konghucu',
            'Lainnya'
        ];

        return view('admin.registrasi.edit', compact(
            'registrasi',
            'pelatihanList',
            'agamaOptions'
        ));
    }

    public function update(Request $request, RegistrasiPeserta $registrasi)
    {
        $rules = [
            'pelatihan_id' => ['required', 'exists:pelatihan,id'],

            'nama' => ['required', 'string', 'max:255'],
            'nik' => [
                'required',
                'regex:/^\d+$/',
                'max:50',
                Rule::unique('registrasi_peserta', 'nik')
                    ->ignore($registrasi->id)
                    ->where(
                        fn($q) =>
                        $q->where('pelatihan_id', $request->pelatihan_id)
                    ),
            ],

            'nip' => ['nullable', 'regex:/^\d+$/', 'max:50'],
            'jabatan' => ['nullable', 'string', 'max:150'],
            'pangkat_golongan' => ['nullable', 'string', 'max:150'],

            'agama' => ['required', Rule::in([
                'Islam',
                'Kristen',
                'Katolik',
                'Hindu',
                'Buddha',
                'Konghucu',
                'Lainnya'
            ])],

            'jenis_kelamin' => ['required', 'in:Laki-laki,Perempuan'],
            'tempat_lahir' => ['nullable', 'string', 'max:150'],
            'tanggal_lahir' => ['nullable', 'date'],

            'no_hp' => ['required', 'regex:/^\d+$/', 'max:30'],
            'email_pribadi' => ['required', 'email', 'max:255'],

            'pendidikan_terakhir' => ['nullable', 'string', 'max:100'],
            'prodi' => ['nullable', 'string', 'max:150'],

            // UNIT KERJA
            'unit_kerja' => ['required', 'string', 'max:255'],
            'npsn'       => ['required', 'regex:/^\d+$/', 'max:20'],
            'alamat_unit_kerja' => ['required', 'string'],
            'kab_kota_unit' => ['required', 'string', 'max:100'],
            'provinsi_unit' => ['required', 'string', 'max:100'],
            'kode_pos_unit' => ['required', 'regex:/^\d+$/', 'max:10'],
            'telp_instansi' => ['required', 'regex:/^\d+$/', 'max:30'],

            // ALAMAT RUMAH
            'alamat_rumah' => ['required', 'string'],
            'kab_kota_rumah' => ['required', 'string', 'max:100'],
            'provinsi_rumah' => ['required', 'string', 'max:100'],
            'kode_pos_rumah' => ['required', 'regex:/^\d+$/', 'max:10'],

            // BANK & NPWP
            'npwp' => ['required', 'regex:/^\d+$/', 'max:50'],
            'bank' => ['required', 'string', 'max:100'],
            'no_rekening' => ['required', 'regex:/^\d+$/', 'max:50'],

            'foto' => ['nullable', 'image', 'mimes:jpg,jpeg,png', 'max:2048'],
        ];

        $data = $request->validate($rules);

        // 🔒 jangan izinkan user_id berubah
        unset($data['user_id']);

        // sinkronkan nama kegiatan
        $pelatihan = Pelatihan::findOrFail($data['pelatihan_id']);
        $data['nama_kegiatan'] = $pelatihan->nama_pelatihan;

        // upload foto baru (jika ada)
        if ($request->hasFile('foto')) {
            if ($registrasi->foto && Storage::disk('public')->exists($registrasi->foto)) {
                Storage::disk('public')->delete($registrasi->foto);
            }

            $data['foto'] = $request->file('foto')
                ->store('registrasi_peserta', 'public');
        }

        $registrasi->update($data);

        return redirect()
            ->route('admin.registrasi.index')
            ->with('success', 'Data registrasi berhasil diperbarui.');
    }


    public function destroy(RegistrasiPeserta $registrasi)
    {
        // Optional: hapus file foto
        if ($registrasi->foto && Storage::disk('public')->exists($registrasi->foto)) {
            Storage::disk('public')->delete($registrasi->foto);
        }

        $registrasi->delete();

        return redirect()
            ->route('admin.registrasi.index')
            ->with('success', 'Data registrasi berhasil dihapus.');
    }
}
