<?php

namespace App\Http\Controllers;

use App\Models\SuratTugas;
use App\Models\Pegawai;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class SuratTugasController extends Controller
{
    public function index()
    {
        $data = SuratTugas::latest()->get();
        return view('admin.kegiatan.surat-tugas.index', compact('data'));
    }

    public function create()
    {
        $pegawais = Pegawai::orderBy('jabatan')
            ->orderBy('nama')
            ->get()
            ->groupBy('jabatan');

        return view('admin.kegiatan.surat-tugas.create', compact('pegawais'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_kegiatan' => 'required',
            'nomor_surat' => 'required',
            'tanggal_surat' => 'required|date',
            'tempat' => 'required|string|max:255',
            'pejabat_penandatangan' => 'required',
            'jabatan_penandatangan' => 'required',
            'pegawai_ids' => 'required|array',
        ]);

        $surat = SuratTugas::create($request->only([
            'nama_kegiatan',
            'nomor_surat',
            'tanggal_surat',
            'tempat',
            'pejabat_penandatangan',
            'nip_penandatangan',
            'jabatan_penandatangan',
        ]));

        foreach ($request->pegawai_ids as $pegawaiId) {
            $surat->anggota()->create([
                'pegawai_id' => $pegawaiId,
            ]);
        }

        return redirect()->route('surat-tugas.index')
            ->with('success', 'Surat Tugas berhasil dibuat');
    }

    public function edit(SuratTugas $surat)
    {
        $pegawais = Pegawai::orderBy('jabatan')
            ->orderBy('nama')
            ->get()
            ->groupBy('jabatan');

        $selectedPegawai = $surat->anggota
            ->pluck('pegawai_id')
            ->toArray();

        return view(
            'admin.kegiatan.surat-tugas.edit',
            compact('surat', 'pegawais', 'selectedPegawai')
        );
    }

    public function update(Request $request, SuratTugas $surat)
    {
        $request->validate([
            'nama_kegiatan' => 'required',
            'nomor_surat' => 'required',
            'tanggal_surat' => 'required|date',
            'tempat' => 'required|string|max:255',
            'pejabat_penandatangan' => 'required',
            'jabatan_penandatangan' => 'required',
            'pegawai_ids' => 'required|array',
        ]);

        $surat->update($request->only([
            'nama_kegiatan',
            'nomor_surat',
            'tanggal_surat',
            'tempat',
            'pejabat_penandatangan',
            'nip_penandatangan',
            'jabatan_penandatangan',
        ]));

        // reset anggota
        $surat->anggota()->delete();

    foreach ($request->pegawai_ids as $pegawaiId) {
        $surat->anggota()->create([
            'pegawai_id' => $pegawaiId,
        ]);
    }

    return redirect()
        ->route('surat-tugas.index')
        ->with('success', 'Surat Tugas berhasil diperbarui');
    }

    public function destroy(SuratTugas $surat)
    {
        $surat->anggota()->delete();
        $surat->delete();

        return redirect()->route('surat-tugas.index')
            ->with('success', 'Surat Tugas berhasil dihapus');
    }

    public function cetak($id)
    {
        $surat = SuratTugas::with('anggota.pegawai')->findOrFail($id);

        $pdf = Pdf::loadView(
            'admin.kegiatan.pdf.surat-tugas',
            compact('surat')
        )->setPaper('A4', 'portrait');

        return $pdf->stream('Surat-Tugas.pdf');
    }
}
